// Copyright 2015-2016 Espressif Systems (Shanghai) PTE LTD
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at

//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "esp32-hal.h"
#include "sdkconfig.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_attr.h"
#include "nvs_flash.h"
#include "nvs.h"
#include "esp_partition.h"
#include "esp_log.h"
#include "pthread.h"
#include <sys/time.h>

//Undocumented!!! Get chip temperature in Farenheit
//Source: https://github.com/pcbreflux/espressif/blob/master/esp32/arduino/sketchbook/ESP32_int_temp_sensor/ESP32_int_temp_sensor.ino
uint8_t temprature_sens_read();

float temperatureRead()
{
    return (temprature_sens_read() - 32) / 1.8;
}

void yield()
{
    vPortYield();
}

portMUX_TYPE microsMux = portMUX_INITIALIZER_UNLOCKED;
static pthread_key_t microsStore=NULL; // Thread Local Storage Handle

void microsStoreDelete(void * storage) {  // release thread local data when task is delete.
    if(storage) free(storage);
}

unsigned long IRAM_ATTR micros()
{
    if (!microsStore) { // first Time Ever thread local not init'd
        portENTER_CRITICAL_ISR(&microsMux);
        pthread_key_create(&microsStore,microsStoreDelete); // create initial holder
        portEXIT_CRITICAL_ISR(&microsMux);
    }
  
    uint32_t *ptr;// [0] is lastCount, [1] is overFlow

    ptr = pthread_getspecific(microsStore); // get address of storage
 
    if(ptr == NULL) { // first time in this thread, allocate mem, init it.  
        portENTER_CRITICAL_ISR(&microsMux);
        ptr = (uint32_t*)malloc(sizeof(uint32_t)*2);
        pthread_setspecific(microsStore,ptr); // store the pointer to this thread's values
        ptr[0] = 0; // lastCount value
        ptr[1] = 0; // overFlow
        portEXIT_CRITICAL_ISR(&microsMux);
    }  

    unsigned long ccount;
    
    portENTER_CRITICAL_ISR(&microsMux);
    __asm__ __volatile__ ( "rsr     %0, ccount" : "=a" (ccount) ); //get cycle count
    if(ccount < ptr[0]) { // overflow occurred
        ptr[1] += UINT32_MAX / CONFIG_ESP32_DEFAULT_CPU_FREQ_MHZ;
    }
    
    ptr[0] = ccount;
    portEXIT_CRITICAL_ISR(&microsMux);

    return ptr[1] + (ccount / CONFIG_ESP32_DEFAULT_CPU_FREQ_MHZ);
}

unsigned long IRAM_ATTR millis()
{
    return xTaskGetTickCount() * portTICK_PERIOD_MS;
}

void delay(uint32_t ms)
{
    vTaskDelay(ms / portTICK_PERIOD_MS);
}

void IRAM_ATTR delayMicroseconds(uint32_t us)
{
    uint32_t m = micros();
    if(us){
        uint32_t e = (m + us);
        if(m > e){ //overflow
            while(micros() > e){
                NOP();
            }
        }
        while(micros() < e){
            NOP();
        }
    }
}

void initVariant() __attribute__((weak));
void initVariant() {}

void init() __attribute__((weak));
void init() {}

void initArduino()
{
    esp_log_level_set("*", CONFIG_LOG_DEFAULT_LEVEL);
    esp_err_t err = nvs_flash_init();
    if(err == ESP_ERR_NVS_NO_FREE_PAGES){
        const esp_partition_t* partition = esp_partition_find_first(ESP_PARTITION_TYPE_DATA, ESP_PARTITION_SUBTYPE_DATA_NVS, NULL);
        if (partition != NULL) {
            err = esp_partition_erase_range(partition, 0, partition->size);
            if(!err){
                err = nvs_flash_init();
            } else {
                log_e("Failed to format the broken NVS partition!");
            }
        }
    }
    if(err) {
        log_e("Failed to initialize NVS! Error: %u", err);
    }
    init();
    initVariant();
}

//used by hal log
const char * IRAM_ATTR pathToFileName(const char * path)
{
    size_t i = 0;
    size_t pos = 0;
    char * p = (char *)path;
    while(*p){
        i++;
        if(*p == '/' || *p == '\\'){
            pos = i;
        }
        p++;
    }
    return path+pos;
}

