import logging
from odoo import models, fields, api, _
from odoo.tools import pycompat
import datetime
from datetime import timedelta
import base64
import os
import psycopg2
import tempfile
import shutil
import unidecode

_logger = logging.getLogger(__name__)
# Try to use pysftp first, fallback to paramiko
try:
    import paramiko
    SFTP_BACKEND = 'paramiko'
except ImportError:
    try:
        import pysftp
        SFTP_BACKEND = 'pysftp'
    except ImportError:
        SFTP_BACKEND = None
        _logger.error("No SFTP library available: install either pysftp or paramiko")

COLUMNS = dict([
('categ', 'Description du groupe de produits'),
('brand', 'Fabricant/marque'),
('product_code', 'Nº d\'article du fabricant'),
('ean', 'EAN'),
('name', 'Description du produit'),
('qty', 'Nombre de ventes'),
('price', 'prix (par article)'),
('package', 'Emballage'),
('unit', 'Unité'),
('medium', 'Domaine'),
('company', 'Société')])

class Company(models.Model):
    _inherit = 'res.company'

    def get_default_gfk_ir_cron_id(self):
        return self.env.ref('sales_stats_export_ftp.ir_cron_sales_stats_scheduler_0', raise_if_not_found=False)

    gfk_sftp_host = fields.Char(
        'SFTP Server',
        help=(
            "The host name or IP address from your remote"
            " server. For example 192.168.0.1"
        )
    )
    gfk_sftp_port = fields.Integer(
        "SFTP Port",
        default=22,
        help="The port on the FTP server that accepts SSH/SFTP calls."
    )
    gfk_sftp_user = fields.Char(
        'Username in the SFTP Server',
        help=(
            "The username where the SFTP connection "
            "should be made with. This is the user on the external server."
        )
    )
    gfk_sftp_password = fields.Char(
        "SFTP Password",
        help="The password for the SFTP connection. If you specify a private "
             "key file, then this is the password to decrypt it.",
    )
    gfk_sftp_folder = fields.Char(
        'SFTP Folder',
        help=(
            "The host name or IP address from your remote"
            " server. For example 192.168.0.1"
        ),
        default="/"
    )
    gfk_category_exclude = fields.Many2many(
        'product.category',
        string="Product Categories to exclude"
    )
    gfk_base_on_ids = fields.Many2many('ir.model',
        domain=[
            ('model','in', ['account.move', 'sale.order', 'pos.order'])],
        string="Based on"
        )
    gfk_brand_field_id = fields.Many2one('ir.model.fields',
        string='Brand Field',
        domain=[('model','=','product.template'), ('ttype','in',['char', 'many2one']), ('modules', '!=', 'product')],
    )
    gfk_file_company_name = fields.Char(
        string='Name',
        help="File Company Name"
    )
    gfk_store_attachment = fields.Boolean(
        string='Store Attachment',
        default=True
    )
    gfk_ir_cron_id = fields.Many2one(
        'ir.cron',
        string='Scheduler',
        default=get_default_gfk_ir_cron_id
    )
    gfk_interval_type = fields.Selection(
        string='Interval',
        related="gfk_ir_cron_id.interval_type"
    )

    def get_pos_group_key(self, pos_line):
        #group_key = str(pos_line.product_id.id)+'pos'+pos_line.order_id.company_id.name
        return 'pos_'+str(pos_line.id)

    def get_brand_name(self, product):
        if self.gfk_brand_field_id and self.gfk_brand_field_id.ttype == 'char':
            return getattr(product, self.gfk_brand_field_id.name)
        elif self.gfk_brand_field_id:
            brand = getattr(product, self.gfk_brand_field_id.name)
            if hasattr(brand, 'display_name'):
                return brand.display_name
            elif hasattr(brand, 'name'):
                return brand.name
        return ''

    def compute_pos_stats(self, lines, to_exclude_categories, product_datas={}):
        self.ensure_one()
        for pos_line in lines:
            product = pos_line.product_id
            if not product:
                continue
            if product.categ_id.id in to_exclude_categories.ids:
                continue
            group_key = self.get_pos_group_key(pos_line)
            if not group_key in product_datas.keys():
                sellers = product.seller_ids.filtered(lambda x:x.product_code)
                product_datas[group_key] = {
                    'categ'       : product.categ_id and product.categ_id.name or '',
                    'brand'       : self.get_brand_name(product),
                    'product_code': sellers and sellers[0].product_code or '',
                    'ean'         : product.barcode and product.barcode or '',
                    'name'        : product.name,
                    'qty'         : pos_line.qty,
                    'subtotal'    : pos_line.price_subtotal_incl,
                    'package'     : '1',
                    'unit'        : pos_line.product_uom_id and pos_line.product_uom_id.name or '',
                    'medium'      : 'pos',
                    'company'     : pos_line.order_id.company_id.name,
                    'pos'         : pos_line.order_id.config_id.name}
            else:
                product_datas[group_key]['qty'] += pos_line.qty
                product_datas[group_key]['subtotal'] += pos_line.price_subtotal
        return product_datas

    def get_invoice_group_key(self, invoice_line):
        #group_key = str(invoice_line.product_id.id)+'invoice'+invoice_line.company_id.name
        return 'invoice_'+str(invoice_line.id)

    def compute_invoices_stats(self, lines, to_exclude_categories, product_datas={}):
        self.ensure_one()
        for invoice_line in lines:
            product = invoice_line.product_id
            if not product:
                continue
            if product.categ_id.id in to_exclude_categories.ids:
                continue
            group_key = self.get_invoice_group_key(invoice_line)
            if not group_key in product_datas.keys():
                sellers = product.seller_ids.filtered(lambda x:x.product_code)
                product_datas[group_key] = {
                    'categ'       : product.categ_id and product.categ_id.name or '',
                    'brand'       : self.get_brand_name(product),
                    'product_code': sellers and sellers[0].product_code or '',
                    'ean'         : product.barcode and product.barcode or '',
                    'name'        : product.name,
                    'qty'         : invoice_line.quantity,
                    'subtotal'    : invoice_line.price_total,
                    'package'     : '1',
                    'unit'        : invoice_line.product_uom_id and invoice_line.product_uom_id.name or '',
                    'medium'      : 'invoice',
                    'company'     : invoice_line.company_id.name}
            else:
                product_datas[group_key]['qty'] += invoice_line.quantity
                product_datas[group_key]['subtotal'] += invoice_line.price_subtotal
        return product_datas

    def get_sales_group_key(self, sale_line):
        #website_installed = self.env['sale.order']._fields.get('website_id', None) != None
        #medium = 'sales'
        #if website_installed and sale_line.order_id.website_id.id:
        #    medium = 'online'
        #group_key = str(sale_line.product_id.id)+medium+sale_line.company_id.name
        return 'sales_'+str(sale_line.id)

    def compute_sales_stats(self, lines, to_exclude_categories, product_datas={}):
        self.ensure_one()
        website_installed = self.env['sale.order']._fields.get('website_id', None) != None
        for sale_line in lines:
            product = sale_line.product_id
            if product.categ_id.id in to_exclude_categories.ids:
                continue
            medium = 'sales'
            if website_installed and sale_line.order_id.website_id.id:
                medium = 'online'
            group_key = self.get_sales_group_key(sale_line)
            if not group_key in product_datas.keys():
                sellers = product.seller_ids.filtered(lambda x:x.product_code)
                medium = 'sales'
                if website_installed and sale_line.order_id.website_id.id:
                    medium = 'online'
                product_datas[group_key] = {
                    'categ'       : product.categ_id and product.categ_id.name or '',
                    'brand'       : self.get_brand_name(product),
                    'product_code': sellers and sellers[0].product_code or '',
                    'ean'         : product.barcode and product.barcode or '',
                    'name'        : product.name,
                    'qty'         : sale_line.product_uom_qty,
                    'subtotal'    : sale_line.price_total,
                    'package'     : '1',
                    'unit'        : sale_line.product_uom and sale_line.product_uom.name or '',
                    'medium'      : medium,
                    'company'     : sale_line.company_id.name}
            else:
                product_datas[group_key]['qty'] += sale_line.product_uom_qty
                product_datas[group_key]['subtotal'] += sale_line.price_subtotal
        return product_datas

    def format_datas_csv(self, product_datas):
        columns = COLUMNS.copy()
        if self.gfk_base_on_ids.filtered(lambda x:x.model == 'pos.order'):
            columns['pos'] = 'Point de vente'
        out = ['"'+'","'.join(columns.values())+'"']
            
        for v in product_datas.values():
            prod_out = []
            for k in columns.keys():
                if v.get(k) != None:
                    prod_out.append(str(v[k]))
                else:
                    prod_out.append('')
            out += ['"'+'","'.join(prod_out)+'"']
        return '\n'.join(out)

    def sftp_connection(self):
        """Return a new SFTP connection with found parameters."""
        self.ensure_one()

        if SFTP_BACKEND == 'pysftp':
            cnopts = pysftp.CnOpts()
            cnopts.hostkeys = None

            params = {
                "host": self.gfk_sftp_host,
                "username": self.gfk_sftp_user,
                "port": self.gfk_sftp_port,
                "cnopts": cnopts,
                "password": self.gfk_sftp_password
            }
            _logger.debug("Trying pysftp connection to sftp://%(username)s@%(host)s:%(port)d", extra=params)
            return pysftp.Connection(**params)

        elif SFTP_BACKEND == 'paramiko':
            _logger.debug("Trying paramiko connection to sftp://%s@%s:%s", self.gfk_sftp_user, self.gfk_sftp_host, self.gfk_sftp_port)
            transport = paramiko.Transport((self.gfk_sftp_host, self.gfk_sftp_port))
            transport.connect(username=self.gfk_sftp_user, password=self.gfk_sftp_password)
            sftp = paramiko.SFTPClient.from_transport(transport)
            sftp._transport = transport  # store reference to close later
            return sftp

        else:
            raise ImportError("No supported SFTP library (pysftp or paramiko) is available")

    def get_stats_content(self, start, stop):
        self.ensure_one()
        product_datas = {}
        to_exclude_categories = self.env['product.category']
        if self.gfk_category_exclude.ids:
            to_exclude_categories = self.env['product.category'].search([('id','child_of', self.gfk_category_exclude.ids)])
        for model in self.gfk_base_on_ids:
            if model.model == 'sale.order':
                sales = self.env['sale.order'].search([('date_order','>=',start),('date_order','<',stop),('state','not in',['draft','sent','cancel'])])
                lines = sales.mapped('order_line').filtered(lambda x:x.product_id and x.product_id.type != 'service')
                self.compute_sales_stats(lines, to_exclude_categories,  product_datas)
            elif model.model == 'account.move':
                invoices = self.env['account.move'].search([('invoice_date','>=',start),('invoice_date','<',stop),('move_type','=','out_invoice')])
                lines = invoices.mapped('line_ids').filtered(lambda x:x.product_id and x.product_id.type != 'service')
                self.compute_invoices_stats(lines, to_exclude_categories, product_datas)
            elif model.model == 'pos.order':
                pos_orders = self.env['pos.order'].search([('date_order','>=',start),('date_order','<',stop), ('state','not in',['draft','cancel'])])
                lines = pos_orders.mapped('lines')
                self.compute_pos_stats(lines, to_exclude_categories, product_datas)
        for k, v in product_datas.items():
            v['price'] = v['qty'] > 1 and int((v['subtotal'] / v['qty'])*100)/100.0 or v['subtotal']
        return self.format_datas_csv(product_datas)

    def action_send_sales_stats(self, date=False):
        for company in self.search([]):
            if not company.gfk_sftp_host:
                continue
            if date and isinstance(date, str):
                date = fields.Date.to_date(date)
            today = date or datetime.datetime.now().date()
            start = today - timedelta(days=today.weekday(), weeks=1)
            stop = start + timedelta(weeks=1)
            company_name = company.gfk_file_company_name
            if not company_name:
                company_name = company.name
            filename = '%s_Gfk_%s.csv' % (unidecode.unidecode(company_name).split(' ')[0].upper(), start.strftime('%YS%V'))
            
            if self.env['ir.attachment'].search([('name','=',filename)]):
                _logger.info('File %s already sent!' % filename)
                continue
            _logger.info('No file %s found' % filename)
            file_content = company.get_stats_content(start, stop)
            datas =  file_content.encode('utf-8')
            remote = company.sftp_connection()
            try:
                # Directory must exist
                if company.gfk_sftp_folder != '/':
                    try:
                        if SFTP_BACKEND == 'pysftp':
                            remote.makedirs(company.gfk_sftp_folder)
                        else:
                            # Manual directory creation for paramiko
                            parts = company.gfk_sftp_folder.strip('/').split('/')
                            path = ''
                            for part in parts:
                                path += '/' + part
                                try:
                                    remote.mkdir(path)
                                except IOError:
                                    pass
                    except Exception as e:
                        _logger.info(f'Error on create remote folder: {e}')

                full_path = os.path.join(company.gfk_sftp_folder, filename)
                with remote.open(full_path, "wb") if SFTP_BACKEND == 'pysftp' else remote.open(full_path, "wb") as f:
                    f.write(datas)

                if company.gfk_store_attachment:
                    self.env['ir.attachment'].create({
                        'datas': base64.b64encode(datas),
                        'name': filename,
                        'store_fname': filename,
                        'mimetype': 'text/csv'
                    })
            finally:
                if SFTP_BACKEND == 'paramiko':
                    remote.close()
                    remote._transport.close()
