import logging
from odoo import models, fields, api, _
from odoo.tools import pycompat
import datetime
from datetime import timedelta
import base64
import os
import psycopg2
import tempfile
import shutil
import unidecode
_logger = logging.getLogger(__name__)
try:
    import pysftp
except ImportError:  # pragma: no cover
    _logger.debug('Cannot import pysftp')

COLUMNS = dict([
('categ', 'Description du groupe de produits'),
('brand', 'Fabricant/marque'),
('product_code', 'Nº d\'article du fabricant'),
('ean', 'EAN'),
('name', 'Description du produit'),
('qty', 'Nombre de ventes'),
('price', 'prix (par article)'),
('package', 'Emballage'),
('unit', 'Unité')])

class Company(models.Model):
    _inherit = 'res.company'

    gfk_sftp_host = fields.Char(
        'SFTP Server',
        help=(
            "The host name or IP address from your remote"
            " server. For example 192.168.0.1"
        )
    )
    gfk_sftp_port = fields.Integer(
        "SFTP Port",
        default=22,
        help="The port on the FTP server that accepts SSH/SFTP calls."
    )
    gfk_sftp_user = fields.Char(
        'Username in the SFTP Server',
        help=(
            "The username where the SFTP connection "
            "should be made with. This is the user on the external server."
        )
    )
    gfk_sftp_password = fields.Char(
        "SFTP Password",
        help="The password for the SFTP connection. If you specify a private "
             "key file, then this is the password to decrypt it.",
    )
    gfk_sftp_folder = fields.Char(
        'SFTP Folder',
        help=(
            "The host name or IP address from your remote"
            " server. For example 192.168.0.1"
        ),
        default="/"
    )
    gfk_category_exclude = fields.Many2many(
        'product.category',
        string="Product Categories to exclude"
    )
    gfk_base_on_ids = fields.Many2many('ir.model',
        domain=[
            ('model','in', ['account.move', 'sale.order', 'pos.order'])],
        string="Based on"
        )
    gfk_brand_field_id = fields.Many2one('ir.model.fields',
        string='Brand Field',
        domain=[('model','=','product.template'), ('ttype','in',['char', 'many2one']), ('modules', '!=', 'product')],
    )
    gfk_store_attachment = fields.Boolean(
        string='Store Attachment',
        default=True
    )

    def compute_pos_stats(self, lines, to_exclude_categories, product_datas={}):
        self.ensure_one()
        for pos_line in lines:
            product = pos_line.product_id
            if not product:
                continue
            if product.categ_id.id in to_exclude_categories.ids:
                continue
            if not product.id in product_datas.keys():
                sellers = product.seller_ids.filtered(lambda x:x.product_code)
                brand_name = ''
                if self.gfk_brand_field_id and self.gfk_brand_field_id.ttype == 'char':
                    brand_name = getattr(product, self.gfk_brand_field_id.name)
                elif self.gfk_brand_field_id:
                    brand_name = getattr(product, self.gfk_brand_field_id.name).name
                product_datas[product.id] = {
                    'categ'       : product.categ_id and product.categ_id.name or '',
                    'brand'       : brand_name and brand_name or '',
                    'product_code': sellers and sellers[0].product_code or '',
                    'ean'         : product.barcode and product.barcode or '',
                    'name'        : product.name,
                    'qty'         : pos_line.qty,
                    'subtotal'    : pos_line.price_subtotal,
                    'package'     : '1',
                    'unit'        : pos_line.product_uom_id and pos_line.product_uom_id.name or ''}
            else:
                product_datas[product.id]['qty'] += pos_line.qty
                product_datas[product.id]['subtotal'] += pos_line.price_subtotal
        return product_datas

    def compute_invoices_stats(self, lines, to_exclude_categories, product_datas={}):
        self.ensure_one()
        for invoice_line in lines:
            product = invoice_line.product_id
            if not product:
                continue
            if product.categ_id.id in to_exclude_categories.ids:
                continue
            if not product.id in product_datas.keys():
                sellers = product.seller_ids.filtered(lambda x:x.product_code)
                brand_name = ''
                if self.gfk_brand_field_id and self.gfk_brand_field_id.ttype == 'char':
                    brand_name = getattr(product, self.gfk_brand_field_id.name)
                elif self.gfk_brand_field_id:
                    brand_name = getattr(product, self.gfk_brand_field_id.name).name
                product_datas[product.id] = {
                    'categ'       : product.categ_id and product.categ_id.name or '',
                    'brand'       : brand_name and brand_name or '',
                    'product_code': sellers and sellers[0].product_code or '',
                    'ean'         : product.barcode and product.barcode or '',
                    'name'        : product.name,
                    'qty'         : invoice_line.quantity,
                    'subtotal'    : invoice_line.price_total,
                    'package'     : '1',
                    'unit'        : invoice_line.product_uom_id and invoice_line.product_uom_id.name or ''}
            else:
                product_datas[product.id]['qty'] += invoice_line.quantity
                product_datas[product.id]['subtotal'] += invoice_line.price_subtotal
        return product_datas

    def compute_sales_stats(self, lines, to_exclude_categories, product_datas={}):
        self.ensure_one()
        for sale_line in lines:
            product = sale_line.product_id
            if product.categ_id.id in to_exclude_categories.ids:
                continue
            if not product.id in product_datas.keys():
                sellers = product.seller_ids.filtered(lambda x:x.product_code)
                if self.gfk_brand_field_id and self.gfk_brand_field_id.ttype == 'char':
                    brand_name = getattr(product, self.gfk_brand_field_id.name)
                elif self.gfk_brand_field_id:
                    brand_name = getattr(product, self.gfk_brand_field_id.name).name
                product_datas[product.id] = {
                    'categ'       : product.categ_id and product.categ_id.name or '',
                    'brand'       : brand_name and brand_name or '',
                    'product_code': sellers and sellers[0].product_code or '',
                    'ean'         : product.barcode and product.barcode or '',
                    'name'        : product.name,
                    'qty'         : sale_line.product_uom_qty,
                    'subtotal'    : sale_line.price_total,
                    'package'     : '1',
                    'unit'        : sale_line.product_uom and sale_line.product_uom.name or ''}
            else:
                product_datas[product.id]['qty'] += sale_line.product_uom_qty
                product_datas[product.id]['subtotal'] += sale_line.price_subtotal
        return product_datas

    def format_datas_csv(self, product_datas):
        out = ['"'+'","'.join(COLUMNS.values())+'"']
        for v in product_datas.values():
            prod_out = []
            for k in COLUMNS.keys():
                prod_out.append(str(v[k]))
            out += ['"'+'","'.join(prod_out)+'"']
        return '\n'.join(out)

    def sftp_connection(self):
        """Return a new SFTP connection with found parameters."""
        self.ensure_one()

        cnopts = pysftp.CnOpts()
        cnopts.hostkeys = None

        params = {
            "host"    : self.gfk_sftp_host,
            "username": self.gfk_sftp_user,
            "port"    : self.gfk_sftp_port,
            "cnopts"  :cnopts
        }
        _logger.debug(
            "Trying to connect to sftp://%(username)s@%(host)s:%(port)d",
            extra=params)
        params["password"] = self.gfk_sftp_password

        return pysftp.Connection(**params)

    def get_stats_content(self, start, stop):
        self.ensure_one()
        product_datas = {}
        to_exclude_categories = self.env['product.category']
        if self.gfk_category_exclude.ids:
            to_exclude_categories = self.env['product.category'].search([('id','child_of', self.gfk_category_exclude.ids)])
        for model in self.gfk_base_on_ids:
            if model.model == 'sale.order':
                sales = self.env['sale.order'].search([('date_order','>=',start),('date_order','<',stop),('state','not in',['draft','sent','calcel'])])
                lines = sales.mapped('order_line').filtered(lambda x:x.product_id and x.product_id.type != 'service')
                self.compute_sales_stats(lines, to_exclude_categories,  product_datas)
            elif model.model == 'account.move':
                invoices = self.env['account.move'].search([('invoice_date','>=',start),('invoice_date','<',stop)])
                lines = invoices.mapped('line_ids').filtered(lambda x:x.product_id and x.product_id.type != 'service')
                self.compute_invoices_stats(lines, to_exclude_categories, product_datas)
            elif model.model == 'pos.order':
                pos_orders = self.env['pos.order'].search([('date_order','>=',start),('date_order','<',stop), ('state','not in',['draft','cancel'])])
                lines = pos_orders.mapped('lines')
                self.compute_pos_stats(lines, to_exclude_categories, product_datas)
        for k, v in product_datas.items():
            v['price'] = int((v['subtotal'] / v['qty'])*100)/100.0
        return self.format_datas_csv(product_datas)

    def action_send_sales_stats(self, date=False):
        for company in self.search([]):
            _logger.info(company.gfk_sftp_host)
            if not company.gfk_sftp_host:
                continue
            today = date or datetime.datetime.now().date()
            start = today - timedelta(days=today.weekday(), weeks=1)
            stop = start + timedelta(weeks=1)
            filename = '%s_Gfk_%s.csv' % (unidecode.unidecode(company.name).split(' ')[0].upper(), start.strftime('%YS%V'))
            
            if self.env['ir.attachment'].search([('name','=',filename)]):
                _logger.info('File %s already sent!' % filename)
                continue
            _logger.info('No file %s found' % filename)
            file_content = company.get_stats_content(start, stop)
            datas =  file_content.encode('utf-8')
            _logger.info(datas)
            with company.sftp_connection() as remote:
                # Directory must exist
                if company.gfk_sftp_folder != '/':
                    try:
                        remote.makedirs(company.gfk_sftp_folder)
                    except pysftp.ConnectionException:
                        _logger.info('Error on create remote folder')
                        pass
                with remote.open(
                        os.path.join(company.gfk_sftp_folder, filename),
                        "wb") as destiny:
                    destiny.write(datas)
                with remote.open(
                        os.path.join(company.gfk_sftp_folder, filename),
                        "r") as destiny:
                    ftp_data = destiny.read()
            if company.gfk_store_attachment:
                self.env['ir.attachment'].create({
                    'datas':base64.b64encode(datas),
                    'name':filename,
                    'store_fname':filename,
                    'mimetype':'text/csv'
                })
